%feature("docstring") OT::QuadraticLeastSquares
R"RAW(Second order polynomial response surface by least squares.

Parameters
----------
dataIn : 2-d sequence of float
    Input data.
dataOut : 2-d sequence of float
    Output data. If not specified, this sample is computed such as:
    :math:`dataOut = h(dataIn)`.

Notes
-----
Instead of replacing the model response :math:`h(\vect{x})` for a *local*
approximation around a given set :math:`\vect{x}_0` of input parameters as in
Taylor approximations, one may seek a *global* approximation of
:math:`h(\vect{x})` over its whole domain of definition. A common choice to
this end is global polynomial approximation.

We consider here a global approximation of the model response using  a
quadratic function:

.. math::

    \vect{y} \, \approx \, \widehat{h}(\vect{x}) \,
                      = \, \sum_{j=0}^{P-1} \; a_j \; \psi_j(\vect{x})

where :math:`P = 1+2n_X +n_X (n_X -1)/2` denotes the number of terms,
:math:`(a_j  \, , \, j=0, \cdots,P-1)` is a set of unknown coefficients and the
family :math:`(\psi_j,j=0,\cdots, P-1)` gathers the constant monomial :math:`1`,
the monomials of degree one :math:`x_i`, the cross-terms :math:`x_i x_j` as
well as the monomials of degree two :math:`x_i^2`. Using the vector notation
:math:`\vect{a} \, = \, (a_{0} , \cdots , a_{P-1} )^{\textsf{T}}` and
:math:`\vect{\psi}(\vect{x}) \, = \, (\psi_0(\vect{x}), \cdots, \psi_{P-1}(\vect{x}) )^{\textsf{T}}`,
this rewrites:

.. math::

    \vect{y} \, \approx \, \widehat{h}(\vect{x}) \,
                      = \, \vect{a}^{\textsf{T}} \; \vect{\psi}(\vect{x})

A *global* approximation of the model response over its whole definition domain
is sought. To this end, the coefficients :math:`a_j` may be computed using a
least squares regression approach. In this context, an experimental design
:math:`\vect{\cX} =(x^{(1)},\cdots,x^{(N)})`, i.e. a set of realizations of
input parameters is required, as well as the corresponding model evaluations
:math:`\vect{\cY} =(y^{(1)},\cdots,y^{(N)})`.

The following minimization problem has to be solved:

.. math::

    \mbox{Find} \quad \widehat{\vect{a}} \quad \mbox{that minimizes}
      \quad \cJ(\vect{a}) \, = \, \sum_{i=1}^N \;
                                \left(
                                y^{(i)} \; - \;
                                \Tr{\vect{a}} \vect{\psi}(\vect{x}^{(i)})
                                \right)^2

The solution is given by:

.. math::

    \widehat{\vect{a}} \, = \, \left(
                               \Tr{\mat{\Psi}} \mat{\Psi}
                               \right)^{-1} \;
                               \Tr{\mat{\Psi}}  \; \vect{\cY}

where:

.. math::

    \mat{\Psi} \, = \, (\psi_{j}(\vect{x}^{(i)}) \; , \; i=1,\cdots,N \; , \; j = 0,\cdots,n_X)

See also
--------
LinearLeastSquares, LinearTaylor, QuadraticTaylor

Examples
--------
>>> import openturns as ot
>>> formulas = ['x1 * sin(x2)', 'cos(x1 + x2)', '(x2 + 1) * exp(x1 - 2 * x2)']
>>> f = ot.SymbolicFunction(['x1', 'x2'], formulas)
>>> X  = [[0.5,0.5], [-0.5,-0.5], [-0.5,0.5], [0.5,-0.5]]
>>> X += [[0.25,0.25], [-0.25,-0.25], [-0.25,0.25], [0.25,-0.25]]
>>> Y = f(X)
>>> algo = ot.QuadraticLeastSquares(X, Y)
>>> algo.run()
>>> mm = algo.getResult().getMetaModel()
>>> x = [0.1, 0.1]
>>> y = mm(x)
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::QuadraticLeastSquares::getInputSample
"Get the input data.

Returns
-------
dataIn : :class:`~openturns.Sample`
    Input data."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuadraticLeastSquares::getConstant
"Get the constant vector of the approximation.

Returns
-------
constantVector : :class:`~openturns.Point`
    Constant vector of the approximation, equal to :math:`a_0`."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuadraticLeastSquares::getOutputSample
"Get the output data.

Returns
-------
dataOut : :class:`~openturns.Sample`
    Output data. If not specified in the constructor, the sample is computed
    such as: :math:`dataOut = h(dataIn)`."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuadraticLeastSquares::setDataOut
"Set the output data.

Parameters
----------
dataOut : 2-d sequence of float
    Output data."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuadraticLeastSquares::getLinear
"Get the linear matrix of the approximation.

Returns
-------
linearMatrix : :class:`~openturns.Matrix`
    Linear matrix of the approximation of the function :math:`h`."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuadraticLeastSquares::getQuadratic
"Get the quadratic term of the approximation.

Returns
-------
tensor : :class:`~openturns.SymmetricTensor`
    Quadratic term of the approximation of the function :math:`h`."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuadraticLeastSquares::getResult
"Get an approximation of the function.

Returns
-------
result : :class:`~openturns.MetaModelResult`
    An approximation of the function :math:`h` by Quadratic Least Squares."

// ---------------------------------------------------------------------

%feature("docstring") OT::QuadraticLeastSquares::run
"Perform the least squares approximation."
