%define OT_OptimizationProblem_doc
R"RAW(Base class to define an optimization problem.

This represents a general optimization problem:

.. math::

    \min_{x\in B} f(x) \\
    g(x) = 0 \\
    h(x) \ge 0

where *B* is problem's bounds, *f* is the objective function, *g* are equality constraints, and *h* are inequality constraints.

Available constructors:
    OptimizationProblem(*objective*)

    OptimizationProblem(*objective, equality, inequality, bounds*)

Parameters
----------
objective : :class:`~openturns.Function`
    Objective function. Additional constraints and bounds must always be
    consistent with the objective input dimension.
equality : :class:`~openturns.Function`
    Equality constraints.
inequality : :class:`~openturns.Function`
    Inequality constraints.
bounds : :class:`~openturns.Interval`
    Bounds.

Examples
--------

Define an optimization problem to find the minimum of the Rosenbrock function:

>>> import openturns as ot
>>> rosenbrock = ot.SymbolicFunction(['x1', 'x2'], ['(1-x1)^2+100*(x2-x1^2)^2'])
>>> problem = ot.OptimizationProblem(rosenbrock)
)RAW"

%enddef

%feature("docstring") OT::OptimizationProblemImplementation
OT_OptimizationProblem_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_getBounds_doc
"Accessor to bounds.

Returns
-------
bounds : :class:`~openturns.Interval`
    Problem bounds."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::getBounds
OT_OptimizationProblem_getBounds_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_getDimension_doc
"Accessor to input dimension.

Returns
-------
dimension : int
    Input dimension of objective function."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::getDimension
OT_OptimizationProblem_getDimension_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_getEqualityConstraint_doc
"Accessor to equality constraints.

Returns
-------
equality : :class:`~openturns.Function`
    Describe equality constraints."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::getEqualityConstraint
OT_OptimizationProblem_getEqualityConstraint_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_getInequalityConstraint_doc
"Accessor to inequality constraints.

Returns
-------
inequality : :class:`~openturns.Function`
    Describe inequality constraints."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::getInequalityConstraint
OT_OptimizationProblem_getInequalityConstraint_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_getLevelFunction_doc
"Accessor to level function.

Returns
-------
level : :class:`~openturns.Function`
    Level function."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::getLevelFunction
OT_OptimizationProblem_getLevelFunction_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_getLevelValue_doc
"Accessor to level value.

Returns
-------
value : float
    Level value."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::getLevelValue
OT_OptimizationProblem_getLevelValue_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_getObjective_doc
"Accessor to objective function.

Returns
-------
objective : :class:`~openturns.Function`
    Objective function."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::getObjective
OT_OptimizationProblem_getObjective_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_hasBounds_doc
"Test whether bounds had been specified.

Returns
-------
value : bool
    *True* if bounds had been set for this problem, *False* otherwise."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::hasBounds
OT_OptimizationProblem_hasBounds_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_hasEqualityConstraint_doc
"Test whether equality constraints had been specified.

Returns
-------
value : bool
    *True* if equality constraints had been set for this problem, *False* otherwise."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::hasEqualityConstraint
OT_OptimizationProblem_hasEqualityConstraint_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_hasInequalityConstraint_doc
"Test whether inequality constraints had been specified.

Returns
-------
value : bool
    *True* if inequality constraints had been set for this problem, *False* otherwise."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::hasInequalityConstraint
OT_OptimizationProblem_hasInequalityConstraint_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_hasLevelFunction_doc
"Test whether level function had been specified.

Returns
-------
value : bool
    *True* if level function had been set for this problem, *False* otherwise."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::hasLevelFunction
OT_OptimizationProblem_hasLevelFunction_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_hasMultipleObjective_doc
"Test whether objective function is a scalar or vector function.

Returns
-------
value : bool
    *False* if objective function is scalar, *True* otherwise."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::hasMultipleObjective
OT_OptimizationProblem_hasMultipleObjective_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_isMinimization_doc
"Test whether this is a minimization or maximization problem.

Parameters
----------
marginal_index : int, default=0
    Index of the output marginal (for multi-objective only)

Returns
-------
value : bool
    *True* if this is a minimization problem (default), *False* otherwise."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::isMinimization
OT_OptimizationProblem_isMinimization_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_setBounds_doc
"Accessor to bounds.

Parameters
----------
bounds : :class:`~openturns.Interval`
    Problem bounds."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::setBounds
OT_OptimizationProblem_setBounds_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_setEqualityConstraint_doc
"Accessor to equality constraints.

Parameters
----------
equalityConstraint : :class:`~openturns.Function`
    Equality constraints."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::setEqualityConstraint
OT_OptimizationProblem_setEqualityConstraint_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_setInequalityConstraint_doc
"Accessor to inequality constraints.

Parameters
----------
inequalityConstraint : :class:`~openturns.Function`
    Inequality constraints."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::setInequalityConstraint
OT_OptimizationProblem_setInequalityConstraint_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_setLevelFunction_doc
"Accessor to level function.

Parameters
----------
levelFunction : :class:`~openturns.Function`
    Level function."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::setLevelFunction
OT_OptimizationProblem_setLevelFunction_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_setLevelValue_doc
"Accessor to level value.

Parameters
----------
levelValue : float
    Level value."

%enddef

%feature("docstring") OT::OptimizationProblemImplementation::setLevelValue
OT_OptimizationProblem_setLevelValue_doc

%define OT_OptimizationProblem_setMinimization_doc
"Tell whether this is a minimization or maximization problem.

Parameters
----------
minimization : bool
    *True* if this is a minimization problem, *False* otherwise.
marginal_index : int, default=0
    Index of the output marginal (for multi-objective only)"
%enddef

%feature("docstring") OT::OptimizationProblemImplementation::setMinimization
OT_OptimizationProblem_setMinimization_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_setObjective_doc
"Accessor to objective function.

Parameters
----------
objectiveFunction : :class:`~openturns.Function`
    Objective function.

Notes
-----
Constraints and bounds are cleared if the objective has a different input
dimension in order to keep the problem valid at all time."
%enddef

%feature("docstring") OT::OptimizationProblemImplementation::setObjective
OT_OptimizationProblem_setObjective_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_hasResidualFunction_doc
"Test whether a least-square problem is defined.

Returns
-------
value : bool
    *True* if this is a least-squares problem, *False* otherwise."
%enddef

%feature("docstring") OT::OptimizationProblemImplementation::hasResidualFunction
OT_OptimizationProblem_hasResidualFunction_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_getResidualFunction_doc
"Accessor to the residual function.

Returns
-------
residualFunction : :class:`~openturns.Function`
    Residual function."
%enddef

%feature("docstring") OT::OptimizationProblemImplementation::getResidualFunction
OT_OptimizationProblem_getLevelFunction_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_setResidualFunction_doc
"Accessor to the residual function.

Parameters
----------
residualFunction : :class:`~openturns.Function`
    Residual function."
%enddef

%feature("docstring") OT::OptimizationProblemImplementation::setResidualFunction
OT_OptimizationProblem_setLevelFunction_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_setVariablesType_doc
"Accessor to the variables type.

Parameters
----------
variablesType : :class:`~openturns.Indices`
    Types of the variables.
    
Notes
-----
Possible values for each variable are `ot.OptimizationProblemImplementation.CONTINUOUS`, `ot.OptimizationProblemImplementation.INTEGER` and `ot.OptimizationProblemImplementation.BINARY`."
%enddef

%feature("docstring") OT::OptimizationProblemImplementation::setVariablesType
OT_OptimizationProblem_setVariablesType_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_getVariablesType_doc
"Accessor to the variables type.

Returns
-------
variablesType : :class:`~openturns.Indices`
    Types of the variables.
    
Notes
-----
Possible values for each variable are `ot.OptimizationProblemImplementation.CONTINUOUS`, `ot.OptimizationProblemImplementation.INTEGER` and ot.OptimizationProblemImplementation.`BINARY`."
%enddef

%feature("docstring") OT::OptimizationProblemImplementation::getVariablesType
OT_OptimizationProblem_getVariablesType_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_isContinuous_doc
"Check if the problem is continuous.

Returns
-------
isContinuous : bool
    Returns True if all variables are continuous."
%enddef

%feature("docstring") OT::OptimizationProblemImplementation::isContinuous
OT_OptimizationProblem_isContinuous_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_isLinear_doc
"Check if the problem is linear.

Returns
-------
isLinear : bool
    Returns True if the problem is of linear type."
%enddef

%feature("docstring") OT::OptimizationProblemImplementation::isLinear
OT_OptimizationProblem_isLinear_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_getLinearCost_doc
"Accessor to the linear cost.

Only relevant for the linear problem type.

Returns
-------
cost : :class:`~openturns.Point`
    Linear cost term.
"
%enddef
%feature("docstring") OT::OptimizationProblemImplementation::getLinearCost
OT_OptimizationProblem_getLinearCost_doc

// ---------------------------------------------------------------------

%define OT_OptimizationProblem_getLinearConstraintCoefficients_doc
"Accessor to the linear constraint term.

Only relevant for the linear problem type.

Returns
-------
A : :class:`~openturns.Matrix`
    Linear inequality constraint term.
"
%enddef
%feature("docstring") OT::OptimizationProblemImplementation::getLinearConstraintCoefficients
OT_OptimizationProblem_getLinearConstraintCoefficients_doc


// ---------------------------------------------------------------------

%define OT_OptimizationProblem_getLinearConstraintBounds_doc
"Accessor to the linear constraint bounds.

Only relevant for the linear problem type.

Returns
-------
LU : :class:`~openturns.Interval`
    Linear inequality constraint bounds.
"
%enddef
%feature("docstring") OT::OptimizationProblemImplementation::getLinearConstraintBounds
OT_OptimizationProblem_getLinearConstraintBounds_doc
