/*
 * openmpt123_sdl2.hpp
 * -------------------
 * Purpose: libopenmpt command line player
 * Notes  : (currently none)
 * Authors: OpenMPT Devs
 * The OpenMPT source code is released under the BSD license. Read LICENSE for more details.
 */

#ifndef OPENMPT123_SDL2_HPP
#define OPENMPT123_SDL2_HPP

#include "openmpt123_config.hpp"
#include "openmpt123.hpp"

#if defined(MPT_WITH_SDL2)

#if defined(__clang__)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wimplicit-fallthrough"
#pragma clang diagnostic ignored "-Wreserved-id-macro"
#endif // __clang__
#include <SDL.h>
#if defined(__clang__)
#pragma clang diagnostic pop
#endif // __clang__
#ifdef main
#undef main
#endif
#ifdef SDL_main
#undef SDL_main
#endif
#if (SDL_COMPILEDVERSION < SDL_VERSIONNUM(2, 0, 4))
MPT_WARNING("Support for SDL2 < 2.0.4 has been deprecated and will be removed in a future openmpt123 version.")
#endif

namespace openmpt123 {

inline constexpr auto sdl2_encoding = mpt::common_encoding::utf8;

struct sdl2_exception : public exception {
private:
	static mpt::ustring text_from_code( int code ) {
		string_concat_stream<mpt::ustring> s;
		s << code;
		return s.str();
	}
public:
	sdl2_exception( int code, const char * error ) : exception( MPT_USTRING("SDL2: ") + text_from_code( code ) + MPT_USTRING(" (") + mpt::transcode<mpt::ustring>( sdl2_encoding, error ? std::string(error) : std::string("NULL") ) + MPT_USTRING(")") ) { }
};

static void check_sdl2_error( int e ) {
	if ( e < 0 ) {
		throw sdl2_exception( e, SDL_GetError() );
	}
}

class sdl2_raii {
public:
	sdl2_raii( Uint32 flags ) {
		check_sdl2_error( SDL_Init( flags ) );
	}
	~sdl2_raii() {
		SDL_Quit();
	}
};

class sdl2_stream_raii : public write_buffers_interface {
private:
	concat_stream<mpt::ustring> & log;
	sdl2_raii sdl2;
	int dev;
	std::size_t channels;
	bool use_float;
	std::size_t sampleQueueMaxFrames;
	std::vector<float> sampleBufFloat;
	std::vector<std::int16_t> sampleBufInt;
protected:
	std::uint32_t round_up_power2(std::uint32_t x)
	{
		std::uint32_t result = 1;
		while ( result < x ) {
			result *= 2;
		}
		return result;
	}
public:
	sdl2_stream_raii( commandlineflags & flags, concat_stream<mpt::ustring> & log_ )
		: log(log_)
		, sdl2( SDL_INIT_NOPARACHUTE | SDL_INIT_TIMER | SDL_INIT_AUDIO )
		, dev(-1)
		, channels(flags.channels)
		, use_float(flags.use_float)
		, sampleQueueMaxFrames(0)
	{
		if ( flags.buffer == default_high ) {
			flags.buffer = 160;
		} else if ( flags.buffer == default_low ) {
			flags.buffer = 80;
		}
		if ( flags.period == default_high ) {
			flags.period = 20;
		} else if ( flags.period == default_low ) {
			flags.period = 10;
		}
		flags.apply_default_buffer_sizes();
		SDL_AudioSpec audiospec;
		std::memset( &audiospec, 0, sizeof( SDL_AudioSpec ) );
		audiospec.freq = flags.samplerate;
		audiospec.format = ( flags.use_float ? AUDIO_F32SYS : AUDIO_S16SYS );
		audiospec.channels = static_cast<Uint8>( flags.channels );
		audiospec.silence = 0;
		audiospec.samples = static_cast<Uint16>( round_up_power2( ( flags.buffer * flags.samplerate ) / ( 1000 * 2 ) ) );
		audiospec.size = static_cast<Uint32>( audiospec.samples * audiospec.channels * ( flags.use_float ? sizeof( float ) : sizeof( std::int16_t ) ) );
		audiospec.callback = NULL;
		audiospec.userdata = NULL;
		if ( flags.verbose ) {
			log << MPT_USTRING("SDL2:") << lf;
			log << MPT_USTRING(" latency: ") << ( audiospec.samples * 2.0 / flags.samplerate ) << MPT_USTRING(" (2 * ") << audiospec.samples << MPT_USTRING(")") << lf;
			log << lf;
		}
		sampleQueueMaxFrames = round_up_power2( ( flags.buffer * flags.samplerate ) / ( 1000 * 2 ) );
		SDL_AudioSpec audiospec_obtained;
		std::memset( &audiospec_obtained, 0, sizeof( SDL_AudioSpec ) );
		std::memcpy( &audiospec_obtained, &audiospec, sizeof( SDL_AudioSpec ) );
		dev = SDL_OpenAudioDevice( NULL, 0, &audiospec, &audiospec_obtained, 0 );
		if ( dev < 0 ) {
			check_sdl2_error( dev );
		} else if ( dev == 0 ) {
			check_sdl2_error( -1 );
		}
		SDL_PauseAudioDevice( dev, 0 );
	}
	~sdl2_stream_raii() {
		SDL_PauseAudioDevice( dev, 1 );
		SDL_CloseAudioDevice( dev );
	}
private:
	std::size_t get_num_writeable_frames() {
		std::size_t num_queued_frames = SDL_GetQueuedAudioSize( dev ) / ( use_float ? sizeof( float ) : sizeof( std::int16_t ) ) / channels;
		if ( num_queued_frames > sampleQueueMaxFrames ) {
			return 0;
		}
		return sampleQueueMaxFrames - num_queued_frames;
	}
	template<typename Tsample>
	void write_frames( const Tsample * buffer, std::size_t frames ) {
		while ( frames > 0 ) {
			std::size_t chunk_frames = std::min( frames, get_num_writeable_frames() );
			if ( chunk_frames > 0 ) {
				check_sdl2_error( SDL_QueueAudio( dev, buffer, static_cast<Uint32>( chunk_frames * channels * ( use_float ? sizeof( float ) : sizeof( std::int16_t ) ) ) ) );
				frames -= chunk_frames;
				buffer += chunk_frames * channels;
			} else {
				SDL_Delay( 1 );
			}
		}
	}
public:
	void write( const std::vector<float*> buffers, std::size_t frames ) override {
		sampleBufFloat.clear();
		for ( std::size_t frame = 0; frame < frames; ++frame ) {
			for ( std::size_t channel = 0; channel < channels; ++channel ) {
				sampleBufFloat.push_back( buffers[channel][frame] );
			}
		}
		write_frames( sampleBufFloat.data(), frames );
	}
	void write( const std::vector<std::int16_t*> buffers, std::size_t frames ) override {
		sampleBufInt.clear();
		for ( std::size_t frame = 0; frame < frames; ++frame ) {
			for ( std::size_t channel = 0; channel < channels; ++channel ) {
				sampleBufInt.push_back( buffers[channel][frame] );
			}
		}
		write_frames( sampleBufInt.data(), frames );
	}
	bool pause() override {
		SDL_PauseAudioDevice( dev, 1 );
		return true;
	}
	bool unpause() override {
		SDL_PauseAudioDevice( dev, 0 );
		return true;
	}
	bool sleep( int ms ) override {
		SDL_Delay( ms );
		return true;
	}
};

inline std::vector<mpt::ustring> show_sdl2_devices( concat_stream<mpt::ustring> & /* log */ ) {
	std::vector<mpt::ustring> devices;
	std::size_t device_index = 0;
	sdl2_raii sdl2( SDL_INIT_NOPARACHUTE | SDL_INIT_AUDIO );
	for ( int driver = 0; driver < SDL_GetNumAudioDrivers(); ++driver ) {
		const char * driver_name = SDL_GetAudioDriver( driver );
		if ( !driver_name ) {
			continue;
		}
		if ( std::string( driver_name ).empty() ) {
			continue;
		}
		if ( SDL_AudioInit( driver_name ) < 0 ) {
			continue;
		}
		for ( int device = 0; device < SDL_GetNumAudioDevices( 0 ); ++device ) {
			string_concat_stream<mpt::ustring> dev;
			const char * device_name = SDL_GetAudioDeviceName( device, 0 );
			if ( !device_name ) {
				continue;
			}
			if ( std::string( device_name ).empty() ) {
				continue;
			}
			dev << device_index << MPT_USTRING(": ") << mpt::transcode<mpt::ustring>( sdl2_encoding, driver_name ) << MPT_USTRING(" - ") << mpt::transcode<mpt::ustring>( sdl2_encoding, device_name );
			device_index++;
			devices.push_back( dev.str() );
		}
		SDL_AudioQuit();
	}
	return devices;
}

inline mpt::ustring show_sdl2_version() {
	string_concat_stream<mpt::ustring> log;
	log << MPT_USTRING("libSDL2 ");
	SDL_version sdlver;
	std::memset(&sdlver, 0, sizeof(SDL_version));
	SDL_GetVersion(&sdlver);
	log << static_cast<int>(sdlver.major) << MPT_USTRING(".") << static_cast<int>(sdlver.minor) << MPT_USTRING(".") << static_cast<int>(sdlver.patch);
	const char* revision = SDL_GetRevision();
	if (revision) {
		log << MPT_USTRING(" (") << mpt::transcode<mpt::ustring>(sdl2_encoding, revision) << MPT_USTRING(")");
	}
	log << MPT_USTRING(", ");
	std::memset(&sdlver, 0, sizeof(SDL_version));
	SDL_VERSION(&sdlver);
	log << MPT_USTRING("API: ") << static_cast<int>(sdlver.major) << MPT_USTRING(".") << static_cast<int>(sdlver.minor) << MPT_USTRING(".") << static_cast<int>(sdlver.patch);
	log << MPT_USTRING(" <https://libsdl.org/>");
	return log.str();
}

} // namespace openmpt123

#endif // MPT_WITH_SDL2

#endif // OPENMPT123_SDL2_HPP

