"""
Compare the generated NumPy code against the NumPy reference code.
"""

import platform

import numpy as np
import pytest
from decimal import Decimal

np.set_printoptions(precision=30)

import gau2grid as gg

_benchmark = {
    0: [[((0, 0, 0), Decimal('1.0'))]],
    1: [[((0, 0, 1), Decimal('1.0'))],
        [((1, 0, 0), Decimal('1.00000000000000000000000000000000000000000000000000000000000'))],
        [((0, 1, 0), Decimal('1.00000000000000000000000000000000000000000000000000000000000'))]],
    2: [[((2, 0, 0), Decimal('-0.50')), ((0, 2, 0), Decimal('-0.50')), ((0, 0, 2), Decimal('1.00'))], [
         ((1, 0, 1), Decimal('1.73205080756887729352744634150587236694280525381038062805581'))],
        [((0, 1, 1), Decimal('1.73205080756887729352744634150587236694280525381038062805581'))],
        [((2, 0, 0), Decimal('0.866025403784438646763723170752936183471402626905190314027905')),
         ((0, 2, 0), Decimal('-0.866025403784438646763723170752936183471402626905190314027905'))],
        [((1, 1, 0), Decimal('1.73205080756887729352744634150587236694280525381038062805581'))]],
    3: [[((2, 0, 1), Decimal('-1.500')), ((0, 2, 1), Decimal('-1.500')), ((0, 0, 3), Decimal('1.000'))],
        [((3, 0, 0), Decimal('-0.612372435695794524549321018676472847991486870164167532108173')),
         ((1, 2, 0), Decimal('-0.612372435695794524549321018676472847991486870164167532108173')),
         ((1, 0, 2), Decimal('2.44948974278317809819728407470589139196594748065667012843269'))],
        [((2, 1, 0), Decimal('-0.612372435695794524549321018676472847991486870164167532108173')),
         ((0, 3, 0), Decimal('-0.612372435695794524549321018676472847991486870164167532108173')),
         ((0, 1, 2), Decimal('2.44948974278317809819728407470589139196594748065667012843269'))],
        [((2, 0, 1), Decimal('1.93649167310370844258963269989119980541646085264579541329378')),
         ((0, 2, 1), Decimal('-1.93649167310370844258963269989119980541646085264579541329378'))], [
         ((1, 1, 1), Decimal('3.87298334620741688517926539978239961083292170529159082658757'))],
        [((3, 0, 0), Decimal('0.790569415042094832999723386108179633429888784831304206714378')),
         ((1, 2, 0), Decimal('-2.37170824512628449899917015832453890028966635449391262014313'))],
        [((2, 1, 0), Decimal('2.37170824512628449899917015832453890028966635449391262014313')),
         ((0, 3, 0), Decimal('-0.790569415042094832999723386108179633429888784831304206714378'))]],
    4: [[((4, 0, 0), Decimal('0.37500')), ((2, 2, 0), Decimal('0.7500')), ((0, 4, 0), Decimal('0.37500')),
         ((2, 0, 2), Decimal('-3.0000')), ((0, 2, 2), Decimal('-3.0000')), ((0, 0, 4), Decimal('1.0000'))],
        [((3, 0, 1), Decimal('-2.37170824512628449899917015832453890028966635449391262014313')),
         ((1, 2, 1), Decimal('-2.37170824512628449899917015832453890028966635449391262014313')),
         ((1, 0, 3), Decimal('3.16227766016837933199889354443271853371955513932521682685750'))],
        [((2, 1, 1), Decimal('-2.37170824512628449899917015832453890028966635449391262014313')),
         ((0, 3, 1), Decimal('-2.37170824512628449899917015832453890028966635449391262014313')),
         ((0, 1, 3), Decimal('3.16227766016837933199889354443271853371955513932521682685750'))],
        [((4, 0, 0), Decimal('-0.559016994374947424102293417182819058860154589902881431067725')),
         ((0, 4, 0), Decimal('0.559016994374947424102293417182819058860154589902881431067725')),
         ((2, 0, 2), Decimal('3.35410196624968454461376050309691435316092753941728858640634')),
         ((0, 2, 2), Decimal('-3.35410196624968454461376050309691435316092753941728858640634'))],
        [((3, 1, 0), Decimal('-1.11803398874989484820458683436563811772030917980576286213545')),
         ((1, 3, 0), Decimal('-1.11803398874989484820458683436563811772030917980576286213545')),
         ((1, 1, 2), Decimal('6.70820393249936908922752100619382870632185507883457717281269'))],
        [((3, 0, 1), Decimal('2.09165006633518886994543006446296872348203842324668049952799')),
         ((1, 2, 1), Decimal('-6.27495019900556660983629019338890617044611526974004149858395'))],
        [((2, 1, 1), Decimal('6.27495019900556660983629019338890617044611526974004149858395')),
         ((0, 3, 1), Decimal('-2.09165006633518886994543006446296872348203842324668049952799'))],
        [((4, 0, 0), Decimal('0.739509972887452005320916036445202131051937653849292540359963')),
         ((2, 2, 0), Decimal('-4.43705983732471203192549621867121278631162592309575524215978')),
         ((0, 4, 0), Decimal('0.739509972887452005320916036445202131051937653849292540359963'))],
        [((3, 1, 0), Decimal('2.95803989154980802128366414578080852420775061539717016143986')),
         ((1, 3, 0), Decimal('-2.95803989154980802128366414578080852420775061539717016143986'))]]
} # yapf: disable


def _test_shell(bench, comp):
    comp_line = sorted(comp)
    bench_line = sorted(bench)
    assert len(comp_line) == len(bench_line)

    for cart in range(len(comp_line)):
        comp_coeff = comp_line[cart]
        bench_coeff = bench_line[cart]

        # Make sure cartesian alignment
        assert comp_coeff[0] == bench_coeff[0]

        # Check coefficient using Decimal tech
        assert comp_coeff[1].quantize(bench_coeff[1]) == bench_coeff[1]

    return True


@pytest.mark.parametrize("AM", _benchmark.keys())
def test_RSH(AM):
    # print("AM %d" % AM)

    pkl_data = gg.RSH.cart_to_RSH_coeffs(AM)
    bench_data = _benchmark[AM]

    assert len(pkl_data) == len(bench_data)
    for sph in range(len(pkl_data)):

        assert _test_shell(bench_data[sph], pkl_data[sph])


def test_RSH_order_p():
    gaus = gg.RSH.cart_to_RSH_coeffs(1, order="gaussian")
    cca = gg.RSH.cart_to_RSH_coeffs(1, order="cca")

    assert _test_shell(gaus[0], cca[1])
    assert _test_shell(gaus[1], cca[2])
    assert _test_shell(gaus[2], cca[0])


def test_RSH_order_d():
    gaus = gg.RSH.cart_to_RSH_coeffs(2, order="gaussian")
    cca = gg.RSH.cart_to_RSH_coeffs(2, order="cca")

    assert _test_shell(gaus[0], cca[2])
    assert _test_shell(gaus[1], cca[3])
    assert _test_shell(gaus[2], cca[1])
    assert _test_shell(gaus[3], cca[4])
    assert _test_shell(gaus[4], cca[0])
